# (c) Copyright 2005-2010. CodeWeavers, Inc.
# This module just handles the MIME part of KDE associations and is shared
# between KDE and Mandrake associations.
package CXMimeKDE;
use warnings;
use strict;

use CXLog;
use CXUtils;
use CXAssoc;
use base "CXAssoc";



#####
#
# MIME database helper functions
#
#####

# In this function we want to get a list of the *native KDE* MIME types
# and their associated extensions. We don't want to include any MIME type
# created by CrossOver in this list.
sub read_mime_db($)
{
    my ($self)=@_;
    return if ($self->{read_mime_db});
    $self->{read_mime_db}=1;

    my @dirs=split /:+/, $self->{mime_path};
    while (@dirs)
    {
        my $dir=shift @dirs;
        my $dh;
        if (!opendir($dh, $dir))
        {
            cxlog("unable to open the '$dir' directory: $!\n");
            next;
        }
        foreach my $dentry (readdir $dh)
        {
            if ($dentry =~ /^\.\.?$/)
            {
                next;
            }
            elsif (-d "$dir/$dentry")
            {
                push @dirs, "$dir/$dentry";
            }
            elsif ($dentry =~ /^x-crossover-/)
            {
                # Ignore CrossOver-specific MIME types
                next;
            }
            elsif ($dentry =~ /\.desktop$/)
            {
                require CXConfig;
                my $mimefile=CXConfig->new("$dir/$dentry");
                my $section=$mimefile->get_section("Desktop Entry") ||
                            $mimefile->get_section("KDE Desktop Entry");
                if (!defined $section)
                {
                    cxlog("'$dir/$dentry' is not a valid .desktop file\n");
                    next;
                }
                my $created_by=$section->get("X-Created-By", "");
                if ($created_by)
                {
                    # Ignore Windows MIME types created by CrossOver
                    cxlog("'$dir/$dentry' is a CrossOver MIME type ($created_by)\n");
                    next;
                }
                my $type=$section->get("Type", "");
                if ($type !~ /^mimetype$/i)
                {
                    cxlog("'$dir/$dentry' is not a MIME type ($type)\n");
                    next;
                }
                my $patterns=$section->get("Patterns", "");
                my @exts=map { s/^\*\.//; $_ } split /;+/, $patterns;

                my $mime=$section->get("MimeType");
                next if (!defined $mime);
                # KDE handles x-executable in a special way so we ignore it
                next if ($mime eq "application/x-executable");
                # KDE only takes the first file it finds into account
                # So don't update the MIME type if it already exists
                next if ($self->mdb_has_mime($mime));
                $self->mdb_add_mime($mime, \@exts);
            }
        }
        closedir($dh);
    }
}


#####
#
# MIME desktop file helper functions
#
#####

sub get_cxmime($$)
{
    my ($self, $filename)=@_;
    my $cxmime=$self->{cxmimes}->{$filename};
    if (!defined $cxmime)
    {
        require CXRWConfig;
        my $desktop=CXRWConfig->new($filename, "", "");
        $cxmime={ desktop    => $desktop };
        if (defined $self->{tag})
        {
            foreach my $domain (@{$self->{domains}})
            {
                my $str=$desktop->get("Desktop Entry", "X-Created-By-$domain-$self->{tag}", "");
                $cxmime->{$domain}={};
                map { $cxmime->{$domain}->{$_}=1 } split /;+/, $str;
            }
        }
        $self->{cxmimes}->{$filename}=$cxmime;
    }
    return $cxmime;
}

sub save_cxmime($$)
{
    my ($self, $cxmime)=@_;
    my $desktop=$cxmime->{desktop};
    my $section=$desktop->get_section("Desktop Entry");
    if (!$section)
    {
        # Clearly not one of our MIME types -> nothing to save
        return 1;
    }

    if ($self->{tag})
    {
        foreach my $domain (@{$self->{domains}})
        {
            if (%{$cxmime->{$domain}})
            {
                $section->set("X-Created-By-$domain-$self->{tag}",
                              join(";", sort keys %{$cxmime->{$domain}}));
            }
            else
            {
                $section->remove("X-Created-By-$domain-$self->{tag}");
            }
        }
    }

    if ($desktop->is_modified())
    {
        my $filename=$desktop->get_filename();
        my $dir=cxdirname($filename);
        if (!cxmkpath($dir))
        {
            cxerr("unable to create the '$dir' directory: $@\n");
            return 0;
        }
        if (!$desktop->save())
        {
            cxerr("unable to save '$filename': $!\n");
            return 0;
        }
    }
    return 1;
}

sub delete_mime_file($)
{
    my ($filename)=@_;

    if (-f $filename)
    {
        cxlog("Deleting '$filename'\n");
        if (!unlink $filename)
        {
            cxerr("unable to delete '$filename': $!\n");
            return 0;
        }
    }

    # Try to delete the parent directory to limit accumulation of cruft
    $filename=cxdirname($filename);
    rmdir $filename;
    return 1;
}

sub create_mime($$$$$$)
{
    my ($self, $domain, $massoc, $mime, $mimetype, $extensions)=@_;
    my $filename="$self->{mime}/$mimetype.desktop";
    my $cxmime=$self->get_cxmime($filename);
    if (!$cxmime->{created})
    {
        cxlog("Creating '$filename'\n");
        my $desktop=$cxmime->{desktop};
        # Recreate the MIME type from scratch...
        my $section=$desktop->get_section("Desktop Entry");
        my $oldsection;
        if ($section)
        {
            my $created_by=$section->get("X-Created-By");
            if (!defined $created_by)
            {
                cxerr("'$filename' does not belong to CrossOver!\n");
                return 0;
            }
            $desktop->remove_all();
            $oldsection=$section;
        }

        $section=$desktop->append_section("Desktop Entry");
        $section->set("Encoding", "UTF-8");
        $section->set("Type", "MimeType");
        $section->set("MimeType", $mimetype);
        CXAssoc::setup_from_best_eassoc($mime);
        # We should really have a description otherwise 'Type:' shows up
        # as an empty string in the file's Properties dialog.
        my $description=$mime->{description} || $mimetype;
        $description.=" (CXMimeKDE)" if ($ENV{CX_TAGALL});
        $section->set("Comment", $description);
        if ($mime->{localize})
        {
            my $oldlang=CXUtils::cxgetlang();
            my $oldencoding=CXUtils::cxsetencoding("UTF-8");
            foreach my $locale (CXUtils::get_supported_locales())
            {
                CXUtils::cxsetlang($locale);
                my $description=cxgettext($mime->{description});
                if ($description ne $mime->{description})
                {
                    $description.=" (CXMimeKDE)" if ($ENV{CX_TAGALL});
                    $section->set("Comment[$locale]", $description);
                }
            }
            CXUtils::cxsetlang($oldlang);
            CXUtils::cxsetencoding($oldencoding);
        }

        $section->set("Icon", $mime->{icon});
        $section->set("Patterns",
                      join(";", map { "*.$_" } sort @$extensions));

        if ($oldsection)
        {
            # ... but preserve the X-Created-By and X-Created-By-* fields
            # from the original to preserve the order so it looks nice
            foreach my $field (@{$oldsection->get_field_list()})
            {
                if ($field =~ /^X-Created-By/)
                {
                    $section->set($field, $oldsection->get($field));
                }
            }
        }
        $section->set("X-Created-By", "CrossOver");

        $cxmime->{created}=1;
    }
    else
    {
        cxlog("Tagging  '$filename'\n");
    }
    $cxmime->{$domain}->{$massoc->{id}}=1;
    return 1;
}

sub query_mime($$$$$)
{
    my ($self, $domain, $massoc, $mimetype, $extensions)=@_;
    my $filename="$self->{mime}/$mimetype.desktop";
    return 0 if (!-f $filename);

    cxlog("Querying '$filename'\n");
    my $cxmime=$self->get_cxmime($filename);
    return 0 if (!$cxmime->{$domain}->{$massoc->{id}});

    my $section=$cxmime->{desktop}->get_section("Desktop Entry");
    return 0 if (!$section);

    my $str=$section->get("MimeType", "");
    return 0 if ($str ne $mimetype);

    my %exts;
    $str=$section->get("Patterns", "");
    map { s/^\*\.//; $exts{$_}=1; } split /;+/, $str;
    foreach my $ext (@$extensions)
    {
        return 0 if (!$exts{$ext});
    }

    return 1;
}

sub untag_mime($$$$;$)
{
    my ($self, $domain, $massoc, $str, $pattern)=@_;
    my $filename=($pattern ? $str : "$self->{mime}/$str.desktop");
    return 1 if (!-f $filename);

    my $cxmime=$self->get_cxmime($filename);
    my $section=$cxmime->{desktop}->get_section("Desktop Entry");
    if (!$section)
    {
        cxlog("unknown format for '$filename'!\n") if (!defined $pattern);
        return 1;
    }

    my $created_by=$section->get("X-Created-By");
    if (!$created_by)
    {
        cxlog("'$filename' does not belong to CrossOver!\n") if (!defined $pattern);
        return 1;
    }

    if (!defined $pattern)
    {
        return 1 if (!$cxmime->{$domain}->{$massoc->{id}});
        cxlog("Untagging '$filename'\n");
        delete $cxmime->{$domain}->{$massoc->{id}};
        return 1 if (%{$cxmime->{$domain}});
        $pattern="$domain-$self->{tag}\$";
    }
    else
    {
        cxlog("Untagging '$filename'\n");
        $cxmime->{$domain}={} if ($self->{tag} and $self->{tag} =~ /^$pattern/);
        $pattern="$domain-$pattern";
    }

    my ($had_tags, $still_has_tags);
    foreach my $field (@{$section->get_field_list()})
    {
        if ($field =~ /^X-Created-By-$pattern/)
        {
            $had_tags=1;
            cxlog("  removing field $field\n");
            $section->remove($field);
        }
        elsif ($field =~ /^X-Created-By-/)
        {
            $still_has_tags=1;
        }
    }
    if (!$had_tags or $still_has_tags)
    {
        cxlog("  not one of ours or still in use\n");
        return 1;
    }

    # No one is using this MIME type anymore
    require CXRWConfig;
    CXRWConfig::uncache_file($filename);
    delete $self->{cxmimes}->{$filename};
    return delete_mime_file($filename);
}



#####
#
# Main
#
#####

my $cxmimekde;
sub get($$$$)
{
    my ($class, $domain, $cxoptions, $gui_info)=@_;
    if (!$cxmimekde)
    {
        $cxmimekde={
            destdir        => $cxoptions->{destdir},
            mime           => "$cxoptions->{destdir}$gui_info->{kde_preferred_mime}",
            mime_path      => $gui_info->{kde_mime_path},
            domains        => []
        };
        bless $cxmimekde, $class;
        $cxmimekde->init_mime_handler($cxoptions);
    }
    if (!grep /^$domain$/, @{$cxmimekde->{domains}})
    {
        push @{$cxmimekde->{domains}}, $domain;
    }
    return $cxmimekde;
}

sub removeall_legacy_mime($$$)
{
    my ($self, $productid, $filename)=@_;

    require CXConfig;
    my $cxmime=CXConfig->new($filename);
    my $section=$cxmime->get_section("Desktop Entry");
    return 1 if (!$section);

    my $created_by=$section->get("X-Created-By");
    if (!$created_by or $created_by !~ /^(CrossOver Office|$productid)$/)
    {
        return 1;
    }
    return delete_mime_file($filename);
}

sub removeall($$)
{
    my ($self, $domain, $pattern)=@_;

    # Scan the MIME database for our MIME types
    my $productid=CXUtils::get_product_id() if ($pattern eq "legacy");
    my @dirs=split /:+/, $self->{mime_path};
    while (@dirs)
    {
        my $dir=$self->{destdir} . shift @dirs;
        next if (!-w $dir);

        my $dh;
        if (!opendir($dh, $dir))
        {
            cxlog("unable to open the '$dir' directory: $!\n");
            next;
        }
        foreach my $dentry (readdir $dh)
        {
            if ($dentry =~ /^\.\.?$/)
            {
                next;
            }
            elsif (-d "$dir/$dentry")
            {
                push @dirs, "$dir/$dentry";
            }
            elsif ($dentry =~ /\.desktop$/)
            {
                if ($pattern eq "legacy")
                {
                    $self->removeall_legacy_mime($productid, "$dir/$dentry");
                }
                else
                {
                    $self->untag_mime($domain, undef, "$dir/$dentry", $pattern);
                }
            }
        }
        closedir($dh);
    }
}

sub finalize($)
{
    my ($self)=@_;
    my $rc=1;
    foreach my $cxmime (values %{$self->{cxmimes}})
    {
        $rc&=$self->save_cxmime($cxmime);
    }
    return $rc;
}

return 1;
